#!/bin/bash

# lowercase Notarize & Deploy Script
# Builds, notarizes, creates DMG, and deploys to maxawad.com/public

set -e

echo "================================================"
echo "  lowercase Notarize & Deploy"
echo "================================================"
echo ""

# Check if notarytool profile is set up
if ! xcrun notarytool history --keychain-profile "notarytool-profile" &>/dev/null; then
    echo "❌ Error: Notarization not configured!"
    echo ""
    echo "Run this once to set up:"
    echo "  xcrun notarytool store-credentials 'notarytool-profile' \\"
    echo "    --apple-id 'your@email.com' \\"
    echo "    --team-id 'J497DX6ZF8' \\"
    echo "    --password 'app-specific-password'"
    echo ""
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$SCRIPT_DIR"
EXPORTER_DIR="$SCRIPT_DIR/../exporter_MacOS"
PUBLIC_DIR="$SCRIPT_DIR/../../../packages/maxawad.com/public"
BUILD_DIR="$PROJECT_DIR/build"
DERIVED_DATA="$BUILD_DIR/DerivedData"

# 1. Build the app (without creating DMG yet)
echo "🔨 Building lowercase.app..."
rm -rf "$BUILD_DIR"
mkdir -p "$BUILD_DIR"

if [ ! -d "$PROJECT_DIR/lowercase.xcodeproj" ]; then
    echo "🔨 Generating Xcode project..."
    cd "$PROJECT_DIR"
    xcodegen generate
fi

cd "$PROJECT_DIR"
xcodebuild \
    -project lowercase.xcodeproj \
    -scheme lowercase \
    -configuration Debug \
    -derivedDataPath "$DERIVED_DATA" \
    clean build

BUILT_APP=$(find "$DERIVED_DATA" -name "lowercase.app" -type d | head -n 1)
if [ ! -d "$BUILT_APP" ]; then
    echo "❌ Error: Built app not found!"
    exit 1
fi

cp -R "$BUILT_APP" "$BUILD_DIR/"
echo "✅ Build complete"
echo ""

# 2. Notarize the app
echo "🔐 Notarizing with Apple (this takes 2-5 minutes)..."
cd "$SCRIPT_DIR"
ditto -c -k --keepParent "$BUILD_DIR/lowercase.app" "lowercase-notarize.zip"

xcrun notarytool submit "lowercase-notarize.zip" \
    --keychain-profile "notarytool-profile" \
    --wait

rm "lowercase-notarize.zip"
echo "✅ Notarization approved!"
echo ""

# 3. Staple the ticket to the app
echo "📎 Stapling notarization ticket..."
xcrun stapler staple "$BUILD_DIR/lowercase.app"
echo "✅ App is now notarized!"
echo ""

# 4. NOW create DMG with notarized app
echo "📦 Creating DMG with notarized app..."
cd "$EXPORTER_DIR"

hdiutil attach ./lowercase-small.dmg -mountpoint /Volumes/lowercase
sleep 2

if [ -d "/Volumes/lowercase/lowercase.app" ]; then
    rm -rf "/Volumes/lowercase/lowercase.app"
fi

cp -R "$BUILD_DIR/lowercase.app" /Volumes/lowercase/
xcrun SetFile -a C "/Volumes/lowercase" 2>/dev/null || true

hdiutil detach /Volumes/lowercase
sleep 2

if [ -f "./lowercase.dmg" ]; then
    rm ./lowercase.dmg
fi

hdiutil convert ./lowercase-small.dmg -format UDZO -o ./lowercase.dmg
echo "✅ DMG created with notarized app"
echo ""

# 5. Deploy to website
if [ -d "$PUBLIC_DIR" ]; then
    cp "./lowercase.dmg" "$PUBLIC_DIR/"
    echo "✅ Deployed to: $PUBLIC_DIR/lowercase.dmg"
    
    FILE_SIZE=$(du -h "$PUBLIC_DIR/lowercase.dmg" | cut -f1)
    echo "📊 File size: $FILE_SIZE"
else
    echo "⚠️  Warning: Public directory not found"
    echo "   DMG available at: $EXPORTER_DIR/lowercase.dmg"
fi

echo ""
echo "================================================"
echo "  ✅ Complete - Ready to Distribute!"
echo "================================================"
echo ""
echo "🌐 Download URL: https://maxawad.com/lowercase.dmg"
echo ""
echo "✨ This DMG will now:"
echo "   • Download without warnings"
echo "   • Show as 'Identified Developer'"
echo "   • Work from ANY location (maxawad.com, gdrive, etc.)"
echo ""
