#!/bin/bash

# lowercase Build & Deploy Script
# This script builds the macOS app, creates a DMG, and deploys to the website

set -e  # Exit on error

echo "================================================"
echo "  lowercase Build & Deploy Script"
echo "================================================"
echo ""

# Define paths
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$SCRIPT_DIR"
EXPORTER_DIR="$SCRIPT_DIR/../exporter_MacOS"
PUBLIC_DIR="$SCRIPT_DIR/../../../packages/maxawad.com/public"
BUILD_DIR="$PROJECT_DIR/build"
DERIVED_DATA="$BUILD_DIR/DerivedData"

# Clean up old builds
echo "🧹 Cleaning previous builds..."
rm -rf "$BUILD_DIR"
mkdir -p "$BUILD_DIR"

# Generate Xcode project from project.yml if needed
if [ ! -d "$PROJECT_DIR/lowercase.xcodeproj" ]; then
    echo "🔨 Generating Xcode project from project.yml..."
    cd "$PROJECT_DIR"
    xcodegen generate
fi

# Build the app
echo "🔨 Building lowercase.app (Debug config for proper signing)..."
cd "$PROJECT_DIR"
xcodebuild \
    -project lowercase.xcodeproj \
    -scheme lowercase \
    -configuration Debug \
    -derivedDataPath "$DERIVED_DATA" \
    clean build

# Find the built app
BUILT_APP=$(find "$DERIVED_DATA" -name "lowercase.app" -type d | head -n 1)

if [ ! -d "$BUILT_APP" ]; then
    echo "❌ Error: Built app not found!"
    exit 1
fi

echo "✅ App built successfully: $BUILT_APP"
echo ""

# Copy to build directory for reference
cp -R "$BUILT_APP" "$BUILD_DIR/"
echo "📦 App copied to: $BUILD_DIR/lowercase.app"
echo ""

# Create DMG
echo "================================================"
echo "  Creating DMG"
echo "================================================"
echo ""

cd "$EXPORTER_DIR"

# Mount the template DMG
echo "📀 Mounting template DMG..."
hdiutil attach ./lowercase-small.dmg -mountpoint /Volumes/lowercase

# Wait for mount
sleep 2

# Remove old app if it exists
if [ -d "/Volumes/lowercase/lowercase.app" ]; then
    echo "🗑️  Removing old app..."
    rm -rf "/Volumes/lowercase/lowercase.app"
fi

# Copy new app
echo "📥 Copying new app to DMG..."
cp -R "$BUILT_APP" /Volumes/lowercase/

# Set custom icon flag
xcrun SetFile -a C "/Volumes/lowercase" 2>/dev/null || true

# Unmount
echo "📤 Ejecting DMG..."
hdiutil detach /Volumes/lowercase

# Wait for unmount
sleep 2

# Remove old final DMG if exists
if [ -f "./lowercase.dmg" ]; then
    echo "🗑️  Removing old final DMG..."
    rm ./lowercase.dmg
fi

# Convert to compressed final DMG
echo "🔨 Creating final compressed DMG..."
hdiutil convert ./lowercase-small.dmg \
  -format UDZO \
  -o ./lowercase.dmg

echo ""
echo "✅ DMG created successfully: lowercase.dmg"
echo ""

# Deploy to website
echo "================================================"
echo "  Deploying to Website"
echo "================================================"
echo ""

if [ -d "$PUBLIC_DIR" ]; then
    echo "🚀 Copying DMG to website public folder..."
    cp "./lowercase.dmg" "$PUBLIC_DIR/"
    echo "✅ Deployed to: $PUBLIC_DIR/lowercase.dmg"
    
    # Get file size
    FILE_SIZE=$(du -h "$PUBLIC_DIR/lowercase.dmg" | cut -f1)
    echo "📊 File size: $FILE_SIZE"
else
    echo "⚠️  Warning: Public directory not found at $PUBLIC_DIR"
    echo "   DMG is available at: $EXPORTER_DIR/lowercase.dmg"
fi

echo ""
echo "================================================"
echo "  ✅ Build Complete!"
echo "================================================"
echo ""
echo "📍 Build artifacts:"
echo "   App:     $BUILD_DIR/lowercase.app"
echo "   DMG:     $EXPORTER_DIR/lowercase.dmg"
echo "   Website: $PUBLIC_DIR/lowercase.dmg"
echo ""
echo "🌐 Download URL: https://maxawad.com/lowercase.dmg"
echo ""
