import Foundation
import SwiftUI
import FluidAudio

enum ModelState {
    case notDownloaded
    case downloading
    case ready
    case error
}

enum ParakeetModelVersion: String, CaseIterable, Identifiable {
    case v2 = "V2 (English)"
    case v3Multilingual = "V3 (Multilingual - 100+ Languages)"
    
    var id: String { rawValue }
    
    var asrVersion: AsrModelVersion {
        switch self {
        case .v2:
            return .v2
        case .v3Multilingual:
            return .v3
        }
    }
    
    var displayName: String { rawValue }
    
    var language: String {
        switch self {
        case .v2:
            return "English Only"
        case .v3Multilingual:
            return "100+ Languages (including English, Russian, Spanish, etc.)"
        }
    }
    
    var size: String {
        switch self {
        case .v2:
            return "~442 MB"
        case .v3Multilingual:
            return "~2.4 GB"
        }
    }
}

@MainActor
class ModelManager: ObservableObject {
    static let shared = ModelManager()
    
    @Published var modelState: ModelState = .notDownloaded
    @Published var downloadProgress: Double = 0
    @Published var errorMessage: String?
    @Published var shouldLowercase: Bool = true  // Default to true - the app is literally called "lowercase"!
    @Published var selectedModelVersion: ParakeetModelVersion = .v2
    
    private init() {
        // Load saved preference
        if let savedVersion = UserDefaults.standard.string(forKey: "selectedModelVersion"),
           let version = ParakeetModelVersion.allCases.first(where: { $0.rawValue == savedVersion }) {
            selectedModelVersion = version
        }
    }
    
    func saveModelPreference() {
        UserDefaults.standard.set(selectedModelVersion.rawValue, forKey: "selectedModelVersion")
    }
}
